// ==============================================================================================================
// Pseudo-Random Dice for ATtiny25/45/85 microcontroller
// Created by Falko Schmidt, University of Ulm, GERMANY
// ==============================================================================================================
#include <Arduino.h>            // General header for Arduino devices / Atmel microcontrollers
#include "DiceDisplay.h"        // The display controller for a dice display using 7 LEDs

// ==============================================================================================================
// Pin definitions for dice display
// PIN_LED_TL_BR     = The pin, which top left (TL) and bottom right (BR) LEDs are connected to
// PIN_LED_TR_BL     = The pin, which top right (TR) and bottom left (BL) LEDs are connected to
// PIN_LED_MID_ROW   = The pin, which the LEDs left and right of the mid row are connected to
// PIN_LED_MID_POINT = The pin, which the single middle LED is connected to
// ==============================================================================================================
#define         PIN_LED_TL_BR             (PIN_PB1)
#define         PIN_LED_TR_BL             (PIN_PB2)
#define         PIN_LED_MID_ROW           (PIN_PB0)
#define         PIN_LED_MID_POINT         (PIN_PB4)

// ==============================================================================================================
// Definitions for the button that will be pressed to roll the dice
// PIN_BUTTON_INPUT  = The pin which the button is connected to
// BUTTON_PRESSED    = The state which the button has, when pressed. (Either HIGH or LOW)
// ==============================================================================================================
#define         PIN_BUTTON_INPUT          (PIN_PB3)

// ==============================================================================================================
// Definitions for delays
// DELAY_CYCLE          = The delay time between two executions of the loop-function
// ROLL_DISPLAY_COUNER  = This variable * DELAY_CYCLE = total result display time in ms.
// ==============================================================================================================
#define         DELAY_CYCLE               (50)
#define         ROLL_DISPLAY_COUNTER      (200)

// ==============================================================================================================
// Initialization of all objects needed.
// eepromHandler    -> instance of class handling load and save operations on EEPROM
// randomGenerator  -> the linear congruential generator used for pseudo-random values
// diceDisplay      -> instance of class controlling dice display LEDs
//
// NOTE: EEPROMHandler must be initialized before randomGenerator, because the randomGenerator uses a
//       seed read from EEPROM
// ==============================================================================================================
DiceDisplay diceDisplay(PIN_LED_TL_BR, PIN_LED_TR_BL, PIN_LED_MID_ROW, PIN_LED_MID_POINT);

// ==============================================================================================================
// Variables needed for program execution
// ==============================================================================================================
const uint8_t shuffleEffectValues[] = {1, 6, 3, 6, 6, 1, 5, 3};
uint8_t shuffleEffectIndex = 0;   // The value shown when button is pressed. Start value is 1
uint8_t delayCounter = 0;         // Delay counter which is used when a number was rolled
bool buttonWasPressed = false;    // boolean indicating if button was pressed before. This indicates a roll
bool isFirstRoll = true;          // boolean indicating if the current roll is the first. Used for new seed value

// ==============================================================================================================
// RANDOM NUMBER GENERATOR
// The code below implements a linear congruential generator for integer values with 64 bits.
// ==============================================================================================================
uint64_t seed;

uint8_t getNextDiceValue() {
    constexpr const uint64_t DICE_VALUE_RANGE_FACTOR = (UINT64_MAX/6);
    seed = seed * 1103515245 + 12345;
    return seed / DICE_VALUE_RANGE_FACTOR + 1;
}

// ==============================================================================================================
// Small inline function for better readability. This function checks if the button is currently pressed
// ==============================================================================================================
inline bool isButtonPressed() {
    return digitalRead(PIN_BUTTON_INPUT) == LOW;
}

// ==============================================================================================================
// This function will be executed once at the beginning of the program execution.
// In this case only the pin, where the external push button is connected to, hast to be defined to be an
// interrupt with internal pull-up.
// ==============================================================================================================
void setup() {
    // Define button pin to be an input with internal pullup
    pinMode(PIN_BUTTON_INPUT, INPUT_PULLUP);
}

// ==============================================================================================================
// This function will be repeated (forever) after execution of the setup function.
// The status of the button is checked. If pressed, a shuffle effect will be displayed. If released the
// rolled value will be displayed for a certain time.
// ==============================================================================================================
void loop() {
  if(isButtonPressed()) {
        // ----------------------------------------------------------------------------------------------------------
        // Button PRESSED -> shuffle effect displayed
        // ----------------------------------------------------------------------------------------------------------
        // Reset values
        delayCounter = 0;
        buttonWasPressed = true;

        // Shuffle effect. We will display the numbers 1...6
        diceDisplay.display(shuffleEffectValues[shuffleEffectIndex]);

        // Update the shuffle effect value.
        if(shuffleEffectIndex == (sizeof(shuffleEffectValues) / sizeof(shuffleEffectValues[0]))) {
          shuffleEffectIndex = 0;
        } else {
          shuffleEffectIndex++;
        }
    } else {

        if(delayCounter == ROLL_DISPLAY_COUNTER) {
            delayCounter = 0;
            diceDisplay.clear();
        } else if(buttonWasPressed) {
          if(isFirstRoll) {
            isFirstRoll = false;
            seed = millis();
          }
          // Show random number
          diceDisplay.display(getNextDiceValue());
          buttonWasPressed = false;
        } else {
          delayCounter++;
        }
    }

    // Wait some time
    delay(DELAY_CYCLE);
}

